<?php

namespace App\Http\Controllers;

use Session;
use App\Category;
use App\Post;
use App\Tag;
use Carbon\Carbon;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Intervention\Image\Facades\Image;
use Illuminate\Support\Facades\Storage;

class PostController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $posts = Post::orderBy('created_at', 'DESC')->paginate(20);
        return view('admin.post.index', compact('posts'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $tags = Tag::all();
        $categories = Category::all();
        return view('admin.post.create', compact(['categories', 'tags']));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */


public function store(Request $request)
{
    // Validate incoming request
    $this->validate($request, [
        'title' => 'required|unique:posts,title',
        'image' => 'required|image',
        'description' => 'required',
        'seotitle' => 'required',
        'seodes' => 'required',
        'posttags' => 'required',
        'category' => 'required',
    ]);

    // Generate slug from title
    $slug = Str::slug(Str::limit($request->title, 50));

    // Prepare image filename
    $image_new_name = $slug . '-' . time() . '.' . $request->file('image')->getClientOriginalExtension();

    // Store the image
    Storage::disk('custom_public')->putFileAs('post', $request->file('image'), $image_new_name);

    // Create post with image path
    $post = Post::create([
        'title' => $request->title,
        'slug' => $slug,
        'image' => 'custom-storage/post/' . $image_new_name,
        'description' => $request->description,
        'seotitle' => $request->seotitle,
        'seodes' => $request->seodes,
        'posttags' => $request->posttags,
        'category_id' => $request->category,
        'user_id' => auth()->user()->id,
        'published_at' => Carbon::now(),
    ]);

    // Attach tags if present
    if ($request->has('tags')) {
        $post->tags()->attach($request->tags);
    }

    // Success message
    Session::flash('success', 'Post created successfully');
    return redirect()->back();
}

    /**
     * Display the specified resource.
     *
     * @param  \App\Post  $post
     * @return \Illuminate\Http\Response
     */
    public function show(Post $post)
    {
        return view('admin.post.show', compact('post'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Post  $post
     * @return \Illuminate\Http\Response
     */
    public function edit(Post $post)
    {
        $tags = Tag::all();
        $categories = Category::all();
        return view('admin.post.edit', compact(['post', 'categories', 'tags']));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Post  $post
     * @return \Illuminate\Http\Response
     */



public function update(Request $request, Post $post)
{
    // Validate request
    $this->validate($request, [
        'title' => "required|unique:posts,title,{$post->id}",
        'description' => 'required',
        'seotitle' => 'required',
        'seodes' => 'required',
        'posttags' => 'required',
        'category' => 'required',
    ]);

    // Update post fields
    $post->title = $request->title;
    $post->slug = Str::slug($request->title);
    $post->description = $request->description;
    $post->seotitle = $request->seotitle;
    $post->seodes = $request->seodes;
    $post->posttags = $request->posttags;
    $post->category_id = $request->category;

    // Sync related tags if applicable
    if ($request->has('tags')) {
        $post->tags()->sync($request->tags);
    }

    // Handle image upload with title-based filename
    if ($request->hasFile('image')) {
        $image = $request->file('image');

        // Optional: delete old image if exists
        if ($post->image && Storage::disk('custom_public')->exists(str_replace('custom-storage/', '', $post->image))) {
            Storage::disk('custom_public')->delete(str_replace('custom-storage/', '', $post->image));
        }

        // Generate slug-based filename
        $slug = Str::slug(Str::limit($request->title, 50));
        $image_new_name = $slug . '-' . time() . '.' . $image->getClientOriginalExtension();

        // Store the file
        Storage::disk('custom_public')->putFileAs('post', $image, $image_new_name);

        // Save path to post
        $post->image = 'custom-storage/post/' . $image_new_name;
    }

    // Save post
    $post->save();

    // Redirect with success message
    Session::flash('success', 'Post updated successfully');
    return redirect()->back();
}



    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Post  $post
     * @return \Illuminate\Http\Response
     */
    public function destroy(Post $post)
    {
        if($post){
            if(file_exists(public_path($post->image))){
                unlink(public_path($post->image));
            }

            $post->delete();
            Session::flash('Post deleted successfully');
        }

        return redirect()->back();
    }



public function singletagdata($title)
{
    $tag = str_replace('-', ' ', $title);
    $tag = ucwords($tag);

    $datalist = DB::table('posts')
        ->where('posttags', 'LIKE', '%' . $tag . '%')
        ->orderBy('id', 'desc') // 'posts.' is not needed because we're querying a single table
        ->get();

    return view('website.post', compact('tag', 'datalist')); // Pass 'tag' and 'datalist' to the view
}


}
